<?php

namespace App\Http\Controllers;

use App\Models\Shop;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Log;


class ShopifyController extends Controller
{
    public function entryPoint(Request $request)
    {
        $shop = $request->get('shop');

        if (!$shop) {
            return response('Shop parameter missing.', 400);
        }

        $shopExists = Shop::where('shopify_domain', $shop)->exists();

        return $shopExists
            ? redirect()->route('shopify.dashboard', ['shop' => $shop])
            : redirect()->route('shopify.install', ['shop' => $shop]);
    }
    /**
     * Redirects the user to Shopify OAuth screen.
     */
    public function install(Request $request)
    {
        // Log info
        Log::info("Shopify request received");

        $shop = $request->get('shop');
        

        if(!$shop) {
            return response("Missing shop parametersssss", 400);
        }

        $API_KEY = env('SHOPIFY_API_KEY');
        
        $scopes = env('SHOPIFY_SCOPES');
        $redirect_uri = env('APP_URL') . env('SHOPIFY_REDIRECT_URI');
        $nonce = bin2hex(random_bytes(12));
        $access_mode = 'per-user';
        
        $oauth_url = 'https://'.$shop.'/admin/oauth/authorize?client_id='.$API_KEY.'&scope='.$scopes.'&redirect_uri='.urlencode($redirect_uri).'&state='.$nonce.'&grant_options[]='.$access_mode;
        
  
        
        return redirect($oauth_url);
    }

    /**
     * Handles Shopify OAuth callback and stores access token and shop info.
     */
    public function callback(Request $request)
    {
        
        $shop = $request->get('shop');
        $code = $request->get('code');
        $hmac = $request->get('hmac');
        
        if (!$shop || !$code) {
            return response("Missing shop or code", 400);
        }
        
         // ✅ validate HMAC
        $params = $request->except('hmac');
        ksort($params);
        $queryString = urldecode(http_build_query($params));
        $calculatedHmac = hash_hmac('sha256', $queryString, env('SHOPIFY_API_SECRET'));
        if (!hash_equals($calculatedHmac, $hmac)) {
            return response("Invalid HMAC", 400);
        }

       
      
        $tokenResponse = Http::asForm()->post("https://{$shop}/admin/oauth/access_token", [
            'client_id' => env('SHOPIFY_API_KEY'),
            'client_secret' => env('SHOPIFY_API_SECRET'),
            'code' => $code,
        ]);

        if (!$tokenResponse->ok()) {
            return response("Failed to get access token", 500);
        }

        $accessToken = $tokenResponse->json()['access_token'];

        // Get shop info
        $shopResponse = Http::withHeaders([
            'X-Shopify-Access-Token' => $accessToken,
        ])->get("https://{$shop}/admin/api/2023-10/shop.json");

        if (!$shopResponse->ok()) {
            return response("Failed to fetch shop info", 500);
        }

        $shopData = $shopResponse->json()['shop'];

        // Save to database
        Shop::updateOrCreate(
            ['shopify_domain' => $shop],
            [
                'access_token' => $accessToken,
                'shop_name' => $shopData['name'],
                'email' => $shopData['email'] ?? null,
            ]
        );
        
        return redirect(env('APP_URL') . "/dashboard?shop={$shop}");

        
    }

    /**
     * Displays the dashboard with shop info.
     */
    public function dashboard(Request $request)
    {
        //$shop = Session::get('shop');
        $shop = $request->get('shop');

        $shopData = Shop::where('shopify_domain', $shop)->first();
        
        if (!$shopData) {
            return redirect()->route('shopify.install', ['shop' => $shop]);
        }


        return view('shopify.dashboard', compact('shopData'));
    }
    
    public function saveEmail(Request $request)
    {
        $shop = $request->input('shop');
        $email = $request->input('email');
        
        $shopData = Shop::where('shopify_domain', $shop)->first();
        
        $shop_name = isset($shopData->shopify_domain) ? $shopData->shopify_domain : '';
        $accessToken = isset($shopData->access_token) ? $shopData->access_token : '';
        
        
        $shopMetaData = DB::table('store_meta_data')
                    ->where('user_email', $email)
                    ->first();
                    
        $hero_banner_title = isset($shopMetaData->hero_banner_title) ? $shopMetaData->hero_banner_title : "Unleash Your Inner Strength!";
                    
        // upload theme 
        $themeData = [
            "theme" => [
                "name" => "My Custom API Theme",
                "src" => "https://performantcode.io/public/shopify_app/themes/my_custom_theme.zip", // Must be a public HTTPS URL
                "role" => "unpublished" // use "unpublished" if you don't want to publish immediately (main = published)
            ]
        ];
        
        $ch = curl_init("https://{$shop_name}/admin/api/2025-07/themes.json");
        
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "X-Shopify-Access-Token: $accessToken",
            "Content-Type: application/json"
        ]);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($themeData));
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $namespace = "custom";
        $key = "banner_title1";
        
        $url = "https://$shop_name/admin/api/2024-07/metafields.json";


        $payload = [
            "metafield" => [
                "namespace" => $namespace,
                "key" => "banner_title1",
                "value" => $hero_banner_title,
                "type" => "single_line_text_field",
                "owner_resource" => "shop" // This is the correct way for REST API
                // No owner_id needed for shop-level metafields in REST
            ]
        ];
    
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Content-Type: application/json",
            "X-Shopify-Access-Token: $accessToken"
        ]);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        
        if ($httpCode >= 200 && $httpCode < 300) {
            $result = json_decode($response, true);
           
        } else {
        
        }
        
        curl_close($ch);
        
        
        return response()->json([
            'message' => 'Data saved successfully'.$response,
            'status' => 'success',
        ], 200);
    }
    
    private function verifyShopifyRequest(Request $request): bool
    {
        $hmacHeader = $request->header('X-Shopify-Hmac-Sha256');
        $data = $request->getContent();
        $calculatedHmac = base64_encode(hash_hmac('sha256', $data, env('SHOPIFY_API_SECRET'), true));
    
        return hash_equals($hmacHeader, $calculatedHmac);
    }

    
    public function requestCustomerData(Request $request)
    {
        return response('done', 200);
    }
    
    public function deleteCustomerData(Request $request)
    {
        return response('done', 200);
    }
    
    public function deleteShopData(Request $request)
    {
        
        $rawPayload = file_get_contents('php://input');
       
        $hmacHeader = $_SERVER['HTTP_X_SHOPIFY_HMAC_SHA256'] ?? '';
        
       
        $calculatedHmac = base64_encode(
            hash_hmac('sha256', $rawPayload, getenv('SHOPIFY_API_SECRET'), true)
        );
        
       
        if (!hash_equals($hmacHeader, $calculatedHmac)) {
            // Invalid → reject request
            return response('Unauthorized', 401);
        }
         
    
        $shop = $request->get('shop');
        
        Shop::where('shopify_domain', $shop)->delete();
        
        return response('success', 200);
    }
    
    public function registerShopWebhook(Request $request)
    {
        echo 'yessssssssssssssss'; die;
        
    }
}
